<?php

namespace Noptin\Addons_Pack\Preferences;

/**
 * Allows subscribers to manage their subscription preferences.
 *
 * @since 1.0.0
 */

defined( 'ABSPATH' ) || exit;

define( 'NOPTIN_MANAGE_PREFERENCES_VERSION', '2.0.0' );
define( 'NOPTIN_MANAGE_PREFERENCES_FILE', __FILE__ );

/**
 * Adds WordPress user related improvements.
 *
 */
class Main {

	/**
	 * Stores the main component instance.
	 *
	 * @access private
	 * @var    Main $instance The main component instance.
	 * @since  1.0.0
	 */
	private static $instance = null;

	/**
	 * Error in current request.
	 */
	protected $error_message;

	/**
	 * Success in current request.
	 */
	protected $success_message;

	/**
	 * Get active instance
	 *
	 * @access public
	 * @since  1.0.0
	 * @return Main The main component instance.
	 */
	public static function instance() {

		if ( empty( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Init variables.
	 *
	 * @since       1.0.0
	 */
	public function __construct() {
		add_filter( 'noptin_db_schema', array( $this, 'add_manage_preferences_url' ), 11 );
		add_filter( 'noptin_subscriber_get_manage_preferences_url', array( $this, 'filter_manage_profile_url' ), 10, 2 );
		add_action( 'noptin_init', array( $this, 'maybe_add_update_subscriber' ), 100 );
		add_action( 'noptin_page_manage_preferences', array( $this, 'display_manage_subscription_form' ) );
		add_filter( 'noptin_get_settings', array( $this, 'add_options' ) );
		add_shortcode( 'noptin_manage_subscription', array( $this, 'get_manage_subscription_form' ) );
    }

	/**
	 * Add manage preferences URL to subscribers schema.
	 *
	 * @param array $schema
	 * @return array
	 */
	public function add_manage_preferences_url( $schema ) {

		if ( isset( $schema['subscribers'] ) ) {
			$schema['subscribers']['props']['manage_preferences_url'] = array(
				'type'        => 'TEXT',
				'description' => __( "The subscriber's manage preferences URL.", 'noptin-addons-pack' ),
				'is_dynamic'  => true,
				'readonly'    => true,
			);
		}

		return $schema;
	}

	/**
	 * Filters the manage profile URL.
	 *
	 * @param string $url
	 * @param \Hizzle\Noptin\DB\Subscriber $subscriber
	 */
	public function filter_manage_profile_url( $url, $subscriber ) {

		$url = get_noptin_option( 'manage_preferences_url' );

		if ( empty( $url ) || get_noptin_action_url( 'manage_preferences' ) === $url ) {
			$url = get_noptin_action_url(
				'manage_preferences',
				noptin_encrypt(
					wp_json_encode(
						array( 'sid' => $subscriber->get_id() )
					)
				)
			);
		} else {
			$url = add_query_arg( 'noptin_key', $subscriber->get_confirm_key(), $url );
		}

		return $url;
	}

	/**
	 * Displays the manage subscriptions HTML.
	 *
	 * @return string
	 */
	public function display_manage_subscription_form() {

		printf(
			'<h1>%s</h1>',
			esc_html( get_bloginfo( 'name' ) )
		);

		$this->the_manage_subscription_form();
	}

	/**
	 * Generates manage subscriptions HTML.
	 *
	 * @return string
	 */
	public function get_manage_subscription_form() {

		// Generate mark-up.
		ob_start();
		$this->the_manage_subscription_form();
		return ob_get_clean();

	}

	/**
	 * Displays the manage subscriptions form.
	 *
	 * @return string
	 */
	public function the_manage_subscription_form() {

		// Prepare subscriber details.
		$subscriber = noptin_get_subscriber( get_current_noptin_subscriber_id() );
		$status     = $subscriber->get_status();
		$key        = $subscriber->get_confirm_key();

		if ( ! $subscriber->exists() ) {
			$subscriber = false;
		}

		// Generate mark-up.
		ob_start();
		?>
			<style>
				.noptin-manage-subscriptions p {
					padding: 0.1875em;
    				margin: 0 0 0.375em;
				}

				.noptin-actions-page-inner .noptin-manage-subscriptions p {
					padding: 0;
					margin: 0 0 16px;
				}

				.noptin-manage-subscriptions .noptin-label {
					font-weight: 700;
					display: block;
				}

				.noptin-manage-subscriptions .noptin-text {
					margin: 0;
					min-height: 32px;
					width: 100%;
					max-width: 610px;
					display: block;
					box-sizing: border-box;
					height: auto;
					padding: 12px;
				}

				.noptin-manage-subscription-success-div p {
					color: green;
				}

				.noptin-manage-subscription-error-div p {
					color: red;
				}

				.noptin-actions-page-inner .button {
					box-sizing: border-box;
    				padding: 12px;
				}

				.screen-reader-text {
					border: 0;
					clip: rect(1px,1px,1px,1px);
					-webkit-clip-path: inset(50%);
					clip-path: inset(50%);
					height: 1px;
					margin: -1px;
					overflow: hidden;
					padding: 0;
					position: absolute;
					width: 1px;
					word-wrap: normal!important;
				}
			</style>
			<form class="noptin-manage-subscriptions" method="POST" action="<?php echo esc_url_raw( add_query_arg( array() ) ); ?>">

				<?php if ( ! empty( $this->error_message ) ) : ?>
					<div class="noptin-manage-subscription-error-div">
						<p><?php echo esc_html( $this->error_message ); ?></p>
					</div>
				<?php endif; ?>

				<?php if ( ! empty( $this->success_message ) ) : ?>
					<div class="noptin-manage-subscription-success-div">
						<p><?php echo esc_html( $this->success_message ); ?></p>
					</div>
				<?php endif; ?>

				<?php if ( apply_filters( 'noptin_manage_subscriptions_show_status_field', true ) ) : ?>
					<p>
						<label class="noptin-label" for="noptin_field_newsletter_status"><?php esc_html_e( 'Newsletter Subscription', 'noptin-addons-pack' ); ?></label>
						<select class="noptin-text" name="noptin_fields[status]" id="noptin_field_newsletter_status">
							<?php foreach ( noptin_get_subscriber_statuses() as $status_key => $status_name ) : ?>
								<option value="<?php echo esc_attr( $status_key ); ?>" <?php selected( $status, $status_key ); ?>><?php echo esc_html( $status_name ); ?></option>
							<?php endforeach; ?>
						</select>
					</p>
				<?php endif; ?>

				<?php

					foreach ( get_noptin_custom_fields( true ) as $custom_field ) {

						// Display the field.
						$custom_field['wrap_name'] = true;
						$custom_field['show_id']   = true;

						echo '<p>';
						display_noptin_custom_field_input( $custom_field, $subscriber );
						echo '</p>';
					}

					wp_nonce_field( 'noptin-manage-subscription-nonce', 'noptin-manage-subscription-nonce' );
				?>

				<input type="hidden" name="noptin-subscriber-key" value="<?php echo esc_attr( $key ); ?>" />

				<p class="submit">
					<input type="submit" name="submit" class="btn button" value="<?php esc_attr_e( 'Update Preferences', 'noptin-addons-pack' ); ?>" />
				</p>
			</form>
		<?php

	}

	/**
	 * Adds / Updates a subscriber when they submit the update preferences form.
	 *
	 * @return string
	 */
	public function maybe_add_update_subscriber() {

		// Abort if no submission was made.
		if ( empty( $_POST['noptin-manage-subscription-nonce'] ) ) {
			return;
		}

		// Verify nonce.
		if ( ! wp_verify_nonce( $_POST['noptin-manage-subscription-nonce'], 'noptin-manage-subscription-nonce' ) ) {
			$this->error_message = __( 'Could not verify nonce. Please try again.', 'noptin-addons-pack' );
			return;
		}

		if ( empty( $_POST['noptin_fields']['email'] ) || ! is_email( $_POST['noptin_fields']['email'] ) ) {
			$this->error_message = __( 'Missing or invalid email address.', 'noptin-addons-pack' );
			return;
		}

		// Retrieve subscriber by key.
		if ( ! empty( $_POST['noptin-subscriber-key'] ) ) {
			$subscriber = noptin_get_subscriber( $_POST['noptin-subscriber-key'] );

			if ( ! $subscriber->exists() || $subscriber->get( 'confirm_key' ) !== $_POST['noptin-subscriber-key'] ) {
				$subscriber = false;
			}
		}

		// Retrieve subscriber by email.
		if ( empty( $subscriber ) ) {
			$subscriber = noptin_get_subscriber( $_POST['noptin_fields']['email'] );

			if ( ! $subscriber->exists() || $subscriber->get( 'email' ) !== $_POST['noptin_fields']['email'] ) {
				$subscriber = false;
			}
		}

		// Create or update subscriber.
		if ( empty( $subscriber ) ) {
			$result = add_noptin_subscriber(
				array_merge(
					array(
						'source' => 'Manage Preferences',
					),
					$_POST['noptin_fields']
				)
			);
		} else {
			$result = update_noptin_subscriber( $subscriber, $_POST['noptin_fields'] );
		}

		if ( is_string( $result ) ) {
			$this->error_message = $result;
		} elseif ( is_wp_error( $result ) ) {
			$this->error_message = $result->get_error_message();
		} else {
			$this->success_message = __( 'Your changes have been saved', 'noptin-addons-pack' );
		}

	}

	/**
	 * Add default lists option
	 * @since       1.0.0
	 */
	public function add_options( $options ) {

		$options['manage_preferences_url']  = array(
			'el'          => 'input',
			'type'        => 'text',
			'placeholder' => get_noptin_action_url( 'manage_preferences' ),
			'default'     => get_noptin_action_url( 'manage_preferences' ),
			'section'     => 'general',
			'label'       => __( 'Manage Preferences URL', 'noptin-addons-pack' ),
			'description' => __( 'Optional. Enter a custom URL to a page where subscribers can manage their subscriptions. Ensure the page has the [noptin_manage_subscription] shortcode.', 'noptin-addons-pack' ),
		);

		return $options;
    }

}
