<?php

namespace Noptin\Addons_Pack\Tasks;

use \Hizzle\Store\Date_Time;

/**
 * This class handles checkout sessions.
 *
 * @version 1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Task.
 */
class Task extends \Hizzle\Store\Record {

	/**
	 * Get the hook.
	 *
	 * @param string $context What the value is for. Valid values are 'view' and 'edit'.
	 * @return string
	 */
	public function get_hook( $context = 'view' ) {
		return $this->get_prop( 'hook', $context );
	}

	/**
	 * Set the hook.
	 *
	 * @param string $value hook.
	 */
	public function set_hook( $value ) {
		$this->set_prop( 'hook', sanitize_text_field( $value ) );
	}

	/**
	 * Get the status.
	 *
	 * @param string $context What the value is for. Valid values are 'view' and 'edit'.
	 * @return string
	 */
	public function get_status( $context = 'view' ) {
		return $this->get_prop( 'status', $context );
	}

	/**
	 * Set the status.
	 *
	 * @param string $value status.
	 */
	public function set_status( $value ) {
		$this->set_prop( 'status', sanitize_text_field( $value ) );
	}

	/**
	 * Get the args.
	 *
	 * @param string $context What the value is for. Valid values are 'view' and 'edit'.
	 * @return string
	 */
	public function get_args( $context = 'view' ) {
		return $this->get_prop( 'args', $context );
	}

	/**
	 * Set the args.
	 *
	 * @param string $value args.
	 */
	public function set_args( $value ) {
		$this->set_prop( 'args', $value );
	}

	/**
	 * Get the args hash.
	 *
	 * @param string $context What the value is for. Valid values are 'view' and 'edit'.
	 * @return string
	 */
	public function get_args_hash( $context = 'view' ) {
		return $this->get_prop( 'args_hash', $context );
	}

	/**
	 * Set the args hash.
	 *
	 * @param string $value args.
	 */
	public function set_args_hash( $value ) {
		$this->set_prop( 'args_hash', $value );
	}

	/**
	 * Get the date created.
	 *
	 * @param string $context What the value is for. Valid values are 'view' and 'edit'.
	 * @return Date_Time|null
	 */
	public function get_date_created( $context = 'view' ) {
		return $this->get_prop( 'date_created', $context );
	}

	/**
	 * Set the date created.
	 *
	 * @param Date_Time|null|string|int $value The date created.
	 */
	public function set_date_created( $value ) {
		$this->set_date_prop( 'date_created', $value );
	}

	/**
	 * Get the date modified.
	 *
	 * @param string $context What the value is for. Valid values are 'view' and 'edit'.
	 * @return Date_Time|null
	 */
	public function get_date_modified( $context = 'view' ) {
		return $this->get_prop( 'date_modified', $context );
	}

	/**
	 * Set the date modified.
	 *
	 * @param Date_Time|null|string|int $value The date modified.
	 */
	public function set_date_modified( $value ) {
		$this->set_date_prop( 'date_modified', $value );
	}

	/**
	 * Get the date scheduled.
	 *
	 * @param string $context What the value is for. Valid values are 'view' and 'edit'.
	 * @return Date_Time|null
	 */
	public function get_date_scheduled( $context = 'view' ) {
		return $this->get_prop( 'date_scheduled', $context );
	}

	/**
	 * Set the date scheduled.
	 *
	 * @param Date_Time|null|string|int $value The date scheduled.
	 */
	public function set_date_scheduled( $value ) {
		$this->set_date_prop( 'date_scheduled', $value );
	}

	/**
	 * Checks if the task has expired.
	 *
	 * @return bool
	 */
	public function has_expired() {
		$expiration = $this->get_date_scheduled();
		return ! $expiration || $expiration->getTimestamp() < time();
	}

	/**
	 * Returns all logs.
	 *
	 * @return array
	 */
	public function get_logs() {
		$logs = $this->get_meta( 'logs' );
		return is_array( $logs ) ? $logs : array();
	}

	/**
	 * Adds a log message.
	 *
	 * @param string $message Log message.
	 */
	public function add_log( $message ) {
		$logs   = $this->get_logs();
		$logs[] = array(
			'message' => $message,
			'date'    => time(),
		);
		$this->update_meta( 'logs', $logs );
	}

	/**
	 * Runs the task.
	 *
	 * @throws \Exception When the task is invalid.
	 */
	public function run() {

		// Abort if no hook.
		$hook = $this->get_hook();
		if ( empty( $hook ) ) {
			throw new \Exception( 'Invalid task: no hook' );
		}

		$args = json_decode( $this->get_args(), true );

		if ( ! is_array( $args ) ) {
			$args = array();
		}

		// Run the task.
		do_action_ref_array( $hook, $args );
	}

	/**
	 * @param Task $task
	 * @param string $message
	 * @param string $status
	 *
	 * @return string The log entry ID
	 */
	public function set_state( $task, $message, $status = '' ) {
		$task->add_log( $message );

		if ( ! empty( $status ) ) {
			$task->set_status( $status );
		}

		$task->save();
	}

	public function task_canceled( $task ) {
		$this->set_state( $task, 'task canceled', 'canceled' );
	}

	public function task_started( $task ) {
		$this->set_state( $task, 'task started', 'running' );
	}

	public function task_complete( $task ) {
		$this->set_state( $task, 'task complete', 'complete' );
	}

	public function task_failed( $task, $exception ) {
		$this->set_state( $task, sprintf( 'task failed: %s', $exception->getMessage() ), 'failed' );
	}

	public function timed_out( $task, $timeout ) {
		$this->set_state( $task, sprintf( 'task marked as failed after %s seconds. Unknown error occurred. Check server, PHP and database error logs to diagnose cause.', $timeout ), 'failed' );
	}

	public function unexpected_shutdown( $task, $error ) {
		if ( ! empty( $error ) ) {
			$this->set_state( $task, sprintf( 'unexpected shutdown: PHP Fatal error %1$s in %2$s on line %3$s', $error['message'], $error['file'], $error['line'] ), 'failed' );
		}
	}
}
