<?php

namespace Noptin\Addons_Pack\Webhooks;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Registers an automation rule that receives a webhook.
 *
 * @since 1.0.0
 */
class Trigger extends \Noptin_Abstract_Trigger {

	/**
	 * @var array
	 */
	public $current_args = array();

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 * @return string
	 */
	public function __construct() {
		add_action( 'rest_api_init', array( $this, 'register_webhook_endpoint' ) );
		add_filter( 'hizzle_store_rest_prepare_noptin/v1_automation_rules', array( $this, 'add_webhook_url_to_rule' ), 10, 2 );
	}

	/**
	 * @inheritdoc
	 */
	public function get_image() {
		return array(
			'icon' => 'admin-site-alt3',
			'fill' => '#00838f',
		);
	}

	/**
	 * Adds the webhook URL to the rule.
	 *
	 * @param \WP_REST_Response                 $response The response data.
	 * @param \Hizzle\Noptin\DB\Automation_Rule $rule     The automation rule.
	 */
	public function add_webhook_url_to_rule( $response, $rule ) {

		// Ensure this is our trigger.
		if ( $rule->get_trigger_id() === $this->get_id() ) {
			$response->data['metadata'] = array_merge(
				is_array( $response->data['metadata'] ) ? $response->data['metadata'] : array(),
				array(
					'webhook_url' => esc_url( rest_url( 'noptin/v1/webhook/' . noptin_encrypt( $rule->get_id() ) ) ),
				)
			);
		}

		return $response;
	}

	/**
	 * Registers the webhook endpoint.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function register_webhook_endpoint() {

		// Register the webhook endpoint.
		register_rest_route(
			'noptin/v1',
			'/webhook/(?P<noptin_rule_id>[\w\-\.]+)',
			array(
				'args'   => array(
					'noptin_rule_id' => array(
						'description' => __( 'Unique identifier for the automation rule.', 'noptin-addons-pack' ),
						'type'        => 'string',
						'required'    => true,
					),
				),
				array(
					'methods'             => \WP_REST_Server::ALLMETHODS,
					'callback'            => array( $this, 'process_webhook' ),
					'permission_callback' => array( $this, 'verify_webhook' ),
				),
				'schema' => '__return_empty_array',
			)
		);
	}

	/**
	 * Verifies the webhook.
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 * @return WP_Error|boolean
	 * @since 1.0.0
	 * @return bool
	 */
	public function verify_webhook( $request ) {

		// Decrypt the webhook.
		$rule_id = noptin_decrypt( $request['noptin_rule_id'] );

		if ( ! is_numeric( $rule_id ) ) {
			return new \WP_Error( 'noptin_webhook_invalid_rule_id', __( 'Invalid rule id', 'noptin-addons-pack' ), array( 'status' => 400 ) );
		}

        /** @var \Hizzle\Noptin\DB\Automation_Rule $rule */
        $rule = noptin()->db()->get( (int) $rule_id, 'automation_rules' );

        if ( is_wp_error( $rule ) ) {
            return $rule;
        }

		if ( ! $rule->exists() || $this->get_id() !== $rule->get_trigger_id() ) {
			return new \WP_Error( 'noptin_webhook_invalid_rule_id', __( 'Invalid rule id', 'noptin-addons-pack' ), array( 'status' => 400 ) );
		}

		$settings = $rule->get_trigger_settings();

		// Check if specified fields match.
		if ( ! empty( $settings['fields'] ) ) {
			foreach ( $settings['fields'] as $field ) {

				// Skip if not an array.
				if ( ! is_array( $field ) ) {
					continue;
				}

				if ( (string) $request->get_param( $field['key'] ) !== (string) $field['value'] ) {
					return new \WP_Error( 'noptin_webhook_invalid_field', __( 'Invalid field', 'noptin-addons-pack' ), array( 'status' => 400 ) );
				}
			}
		}

		return true;
	}

	/**
	 * @inheritdoc
	 */
	public function get_id() {
		return 'catch_webhook';
	}

	/**
	 * @inheritdoc
	 */
	public function get_name() {
		return __( 'Receive Webhook', 'noptin-addons-pack' );
	}

	/**
	 * @inheritdoc
	 */
	public function get_description() {
		return __( 'When a webhook is received', 'noptin-addons-pack' );
	}

	/**
	 * Retrieve the actions's rule table description.
	 *
	 * @since 1.11.9
	 * @param Noptin_Automation_Rule $rule
	 * @return array
	 */
	public function get_rule_table_description( $rule ) {

		$meta = array(
			__( 'Webhook URL', 'noptin-addons-pack' ) => esc_url( rest_url( 'noptin/v1/webhook/' . noptin_encrypt( $rule->id ) ) ),
		);

		$settings = $rule->trigger_settings;

		// Check if specified fields match.
		if ( ! empty( $settings['fields'] ) ) {
			foreach ( $settings['fields'] as $field ) {

				// Skip if not an array.
				if ( ! is_array( $field ) ) {
					continue;
				}
				// Obfuscate half of the value.
				$length = strlen( $field['value'] );
				$value  = substr( $field['value'], 0, $length / 2 ) . str_repeat( '*', $length / 2 );

				$meta[ $field['key'] ] = $value;
			}
		}

		return $this->rule_trigger_meta( $meta, $rule );
	}

	/**
	 * @inheritdoc
	 */
	public function get_settings() {

		return array(

			'fields' => array(
				'el'          => 'webhook_key_value_repeater',
				'label'       => __( 'Verification', 'noptin-addons-pack' ),
				'description' => __( 'For extra security, you can specify fields and expected values to verify if the webhook is valid.', 'noptin-addons-pack' ),
				'add_field'   => __( 'Add Field', 'noptin-addons-pack' ),
				'default'     => array(),
			),
		);
	}

	/**
     * Returns an array of known smart tags.
     *
     * @since 1.9.0
     * @return array
     */
    public function get_known_smart_tags() {

		$smart_tags = parent::get_known_smart_tags();

		foreach ( $this->current_args as $key => $value ) {
			$smart_tags[ $key ] = array(
				'description'       => $key,
				'example'           => $key,
				'conditional_logic' => is_numeric( $value ) ? 'number' : 'string',
				'replacement'       => $value,
			);
		}

		return $smart_tags;
	}

	/**
	 * Processes the webhook.
	 *
	 * @since 1.0.0
	 * @param \WP_REST_Request $request Full data about the request.
	 * @return void
	 */
	public function process_webhook( $request ) {

		$data  = $this->flatten_array( $request->get_params() );
		$email = isset( $data['email'] ) && is_email( $data['email'] ) ? $data['email'] : '';

		// Retrieve email.
		if ( empty( $email ) ) {
			foreach ( $data as $value ) {
				if ( is_string( $value ) && is_email( $value ) ) {
					$data['email'] = $value;
					$email         = $value;
					break;
				}
			}
		}

		$this->current_args = $data;

		$this->trigger( $email, $data );

		return rest_ensure_response( array( 'success' => true ) );
	}

	/**
	 * Flatten an array.
	 *
	 * @param array $array The array to flatten.
	 * @param string $prefix The prefix to use.
	 * @return array
	 */
	protected function flatten_array( $array, $prefix = '' ) {

		$flattened = array();

		foreach ( $array as $key => $value ) {

			if ( strtolower( $prefix . $key ) === 'email' ) {
				$flattened['email'] = $value;
			}

			$flattened[ $prefix . $key ] = $value;

			if ( is_array( $value ) ) {
				$flattened = array_replace( $flattened, $this->flatten_array( $value, $prefix . $key . '.' ) );
			}
		}

		return $flattened;
	}

}
